<?php

/**
 * Order Controller
 *
 * @package cafejavas
 * @subpackage app\Http\Controllers\Front
 * @author TechAhead
 * @version '1.0'
 * @copyright (c) 2019, CafeJavas
 */

namespace App\Http\Controllers\Front;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;

/**
 * Order Controller
 *
 * @package cafejavas
 * @subpackage app\Http\Controllers\Front
 * @author TechAhead
 * @version '1.0'
 * @copyright (c) 2019, CafeJavas
 */
class OrderController extends Controller {

  /**
   * Function used to get the list of all the orders placed by user
   * @param int $pageNo
   * @return Json
   */
  public function getOrderHistory($pageNo = 1) {
    try {
      if (!ctype_digit($pageNo) || $pageNo == 0) {
        $pageNo = 1;
      }
      $url = "orderHistory";
      $previous = $pageNo - 1;
      $next = $pageNo + 1;
      $userId = getUserId();
      $deviceId = getDeviceId();
      $parameters = array("userId" => $userId, "deviceId" => $deviceId, "pageNo" => $pageNo);
      $response = getRequestAPI('getOrderHistory', $parameters);
      if ($response['Status'] == 400) {
        return view('front.dashboard.error')->with('message', $response['Message']);
      } else {
        return view('front.order.orderHistory')->with(['data' => $response['Result'], "url" => $url, "previous" => $previous, "next" => $next]);
      }
    } catch (Exception $ex) {
      return redirect()->back()->with('error', $ex->getMessage());
    }
  }

  /**
   * Function used to get the details of a order
   * @param Request $request
   * @return Json
   */
  public function getOrderHistoryDetail($id) {
    try {
      $orderId = base64_decode($id);
      $parameters = array("orderId" => $orderId);
      $response = getRequestAPI('getOrderHistoryDetails', $parameters);
      if ($response['Status'] == 400) {
        return view('front.dashboard.error')->with('message', $response['Message']);
      } else {
        return view('front.order.orderHistoryDetail')->with(['data' => $response['Result']]);
      }
    } catch (Exception $ex) {
      return redirect()->back()->with('error', $ex->getMessage());
    }
  }

  /**
   * Function used to show details of a specific order which is completed
   * @param Request $request
   * @return type
   */
  public function specificOrderDetail(Request $request) {

    if ($request->has('data') && $request->data != '') {
      $orderDetail = unserialize($request->data);
      return view('front.order.specificOrderHistoryDetail')->with(['data' => $orderDetail['json']['data']]);
    } else {
      return redirect()->back()->with('error', trans('messages.something_went_wrong'));
    }
  }

  public function trackOrders(Request $request) {
    try {
      $userId = getUserId();
      $deviceId = getDeviceId();
      $parameters = array("userId" => $userId, "deviceId" => $deviceId);
      $response = getRequestAPI('trackOrder', $parameters);
//      dd($response);
      if ($response['Status'] == 400) {
        return view('front.dashboard.error')->with('message', $response['Message']);
      } else {
        return view('front.order.trackOrders')->with(['data' => $response['Result']]);
      }
    } catch (Exception $ex) {
      return redirect()->back()->with('error', $ex->getMessage());
    }
  }

  /**
   * Function used to track the details of a specific order
   * @param type $id
   * @return type
   */
  public function trackOrderDetails($id) {
    try {
      $orderId = base64_decode($id);
      $parameters = array("orderId" => $orderId);
      $response = getRequestAPI('trackOrderdetail', $parameters);
      if ($response['Status'] == 400) {
        return view('front.dashboard.error')->with('message', $response['Message']);
      } else {
        return view('front.order.trackOrderDetail')->with(['data' => $response['Result']]);
      }
    } catch (Exception $ex) {
      return redirect()->back()->with('error', $ex->getMessage());
    }
  }

  /**
   * Function is used to purchase a giftcard for the user
   * @param Request $request
   * @return Illuminate\Http\Response
   */
  public function giftcardPreview(Request $request) {
//    dd($request);
    $rules = array(
        'gift_id' => 'required',
        'gift_amt' => 'required|Integer|min:9999',
        'to_email' => 'required|email',
        'from_email' => 'required|email',
        'from_msg' => 'required'
    );
    $messages = array(
        'gift_amt.required' => 'Please enter the giftcard amount!',
        'gift_amt.Integer' => 'Please enter a valid amount!',
        'gift_amt.min' => 'The gift card amount should be at least 10000!',
        'to_email.required' => 'Please enter the giftcard reciever email!',
        'to_email.email' => 'Please enter a valid reciever email!',
        'from_email.required' => 'Please enter your email here!',
        'from_email.email' => 'Please enter a valid email here!',
        'from_msg.required' => 'Please enter your message here!'
    );
    $validator = Validator::make($request->all(), $rules, $messages);
    if ($validator->fails()) {
      return redirect()->back()->withErrors($validator->errors(), 'giftcardErrors')->withInput($request->all());
    } else {
      // redirect to preview page
      $data['gift_id'] = $request->gift_id;
      $data['gift_amt'] = $request->gift_amt;
      $data['to_email'] = $request->to_email;
      $data['from_email'] = $request->from_email;
      $data['from_msg'] = $request->from_msg;
      return view('front.product.giftcardPreview')->with('data', $data);
    }
  }

  /**
   * Function used to purchase Gift card
   * @param Request $request
   * @return type json
   */
  public function giftcardPurchase(Request $request) {
    if ($request->has('data') && $request->data != '') {
      try {
        $giftDetails = unserialize(base64_decode($request->data));
        $userId = getUserId();
        $isGuestUser = $userId == config('constants.GUEST_USER_ID') ? '1' : '0';
        $parameters = array("userId" => $userId,
            "giftcardId" => explode(",", $giftDetails['gift_id'])[0],
            "isGuestuser" => $isGuestUser, //(0=NO, 1=Yes)
            "userMail" => $giftDetails['from_email'],
            "receiverMail" => $giftDetails['to_email'],
            "price" => $giftDetails['gift_amt'],
            "message" => $giftDetails['from_msg']
        );
        $response = postRequestAPI('giftcardPurchase', $parameters);

        if ($response['Status'] == 200 && $response['Success'] == true) {
          //on success receive the transactionId that is the order id
//          $request->session()->flash('success', "E-Gift card sent!");
          $transactionId = $response['Result']['transactionId'];
//          return response()->json(['success' => $transactionId]);
          //call transaction api from here
          $params = array("isGuestuser" => $isGuestUser, //(0=NO, 1=Yes)
              "itemDesc" => "GiftCard",
              "orderId" => (string) $transactionId, //orderId
//              "totalAmount" => $giftDetails['gift_amt'],
              "totalAmount" => '1001',
              "type" => "2", // (1=Food, 2=giftcard)
              "userId" => $userId
          );
//          return response()->json($params);
          $responseTransaction = postRequestAPI('transaction', $params);
          if ($responseTransaction['Status'] == 200 && $responseTransaction['Success'] == true) {
             return response()->json(['success' => $responseTransaction['Result']]);
          } else {
            return response()->json(['errors' => $responseTransaction['Message']]);
          }
        } else {
          return response()->json(['errors' => $response['Message']]);
        }
      } catch (\Exception $ex) {
        $request->session()->flash('error', $ex->getMessage());
        return response()->json(['error' => $ex->getMessage()]);
      }
    } else {
      return response()->json(['error' => trans('messages.something_went_wrong')]);
    }
  }

}
