"use strict";

module.exports = Iterator;

var Object = require("./shim-object");
var GenericCollection = require("./generic-collection");

// upgrades an iterable to a Iterator
function Iterator(iterable) {

    if (!(this instanceof Iterator)) {
        return new Iterator(iterable);
    }

    if (Array.isArray(iterable) || typeof iterable === "string")
        return Iterator.iterate(iterable);

    iterable = Object(iterable);

    if (iterable instanceof Iterator) {
        return iterable;
    } else if (iterable.next) {
        this.next = function () {
            return iterable.next();
        };
    } else if (iterable.iterate) {
        var iterator = iterable.iterate();
        this.next = function () {
            return iterator.next();
        };
    } else if (Object.prototype.toString.call(iterable) === "[object Function]") {
        this.next = iterable;
    } else {
        throw new TypeError("Can't iterate " + iterable);
    }

}

Iterator.prototype.forEach = GenericCollection.prototype.forEach;
Iterator.prototype.map = GenericCollection.prototype.map;
Iterator.prototype.filter = GenericCollection.prototype.filter;
Iterator.prototype.every = GenericCollection.prototype.every;
Iterator.prototype.some = GenericCollection.prototype.some;
Iterator.prototype.any = GenericCollection.prototype.any;
Iterator.prototype.all = GenericCollection.prototype.all;
Iterator.prototype.min = GenericCollection.prototype.min;
Iterator.prototype.max = GenericCollection.prototype.max;
Iterator.prototype.sum = GenericCollection.prototype.sum;
Iterator.prototype.average = GenericCollection.prototype.average;
Iterator.prototype.flatten = GenericCollection.prototype.flatten;
Iterator.prototype.zip = GenericCollection.prototype.zip;
Iterator.prototype.enumerate = GenericCollection.prototype.enumerate;
Iterator.prototype.sorted = GenericCollection.prototype.sorted;
Iterator.prototype.group = GenericCollection.prototype.group;
Iterator.prototype.reversed = GenericCollection.prototype.reversed;
Iterator.prototype.toArray = GenericCollection.prototype.toArray;
Iterator.prototype.toObject = GenericCollection.prototype.toObject;
Iterator.prototype.iterator = GenericCollection.prototype.iterator;

Iterator.prototype.__iterationObject = null;
Object.defineProperty(Iterator.prototype,"_iterationObject", {
    get: function() {
        return this.__iterationObject || (this.__iterationObject = { done: false, value:void 0});
    }
});


// this is a bit of a cheat so flatten and such work with the generic
// reducible
Iterator.prototype.constructClone = function (values) {
    var clone = [];
    clone.addEach(values);
    return clone;
};

Iterator.prototype.mapIterator = function (callback /*, thisp*/) {
    var self = Iterator(this),
        thisp = arguments[1],
        i = 0;

    if (Object.prototype.toString.call(callback) != "[object Function]")
        throw new TypeError();

    return new self.constructor(function () {
        if(self._iterationObject.done !== true) {
            var callbackValue = callback.call(thisp, self.next().value, i++, self);
            self._iterationObject.value = callbackValue;
        }
        return self._iterationObject;
    });
};

Iterator.prototype.filterIterator = function (callback /*, thisp*/) {
    var self = Iterator(this),
        thisp = arguments[1],
        i = 0;

    if (Object.prototype.toString.call(callback) != "[object Function]")
        throw new TypeError();

    return new self.constructor(function () {
        var nextEntry;
        while (true) {
            nextEntry = self.next();
            if(nextEntry.done !== true) {
                if (callback.call(thisp, nextEntry.value, i++, self))
                    return nextEntry;
            }
            else {
                //done true and value undefined at this point
                return nextEntry;
            }
        }
    });
};

Iterator.prototype.reduce = function (callback /*, initial, thisp*/) {
    var self = Iterator(this),
        result = arguments[1],
        thisp = arguments[2],
        i = 0,
        nextEntry;

    if (Object.prototype.toString.call(callback) != "[object Function]")
        throw new TypeError();

    // first iteration unrolled
    nextEntry = self.next();
    if(nextEntry.done === true) {
        if (arguments.length > 1) {
            return arguments[1]; // initial
        } else {
            throw TypeError("cannot reduce a value from an empty iterator with no initial value");
        }
    }
    if (arguments.length > 1) {
        result = callback.call(thisp, result, nextEntry.value, i, self);
    } else {
        result = nextEntry.value;
    }
    i++;
    // remaining entries
    while (true) {
        nextEntry = self.next();
        if(nextEntry.done === true) {
            return result;
        }
        result = callback.call(thisp, result, nextEntry.value, i, self);
        i++;
    }

};

Iterator.prototype.concat = function () {
    return Iterator.concat(
        Array.prototype.concat.apply(this, arguments)
    );
};

Iterator.prototype.dropWhile = function (callback /*, thisp */) {
    var self = Iterator(this),
        thisp = arguments[1],
        stopped = false,
        stopValue,
        nextEntry,
        i = 0;

    if (Object.prototype.toString.call(callback) != "[object Function]")
        throw new TypeError();

    while (true) {
        nextEntry = self.next();
        if(nextEntry.done === true) {
            break;
        }
        if (!callback.call(thisp, nextEntry.value, i, self)) {
            stopped = true;
            stopValue = nextEntry.value;
            break;
        }
        i++;
    }

    if (stopped) {
        return self.constructor([stopValue]).concat(self);
    } else {
        return self.constructor([]);
    }
};

Iterator.prototype.takeWhile = function (callback /*, thisp*/) {
    var self = Iterator(this),
        thisp = arguments[1],
        nextEntry,
        i = 0;

    if (Object.prototype.toString.call(callback) != "[object Function]")
        throw new TypeError();

    return new self.constructor(function () {
        if(self._iterationObject.done !== true) {
            var value = self.next().value;
            if(callback.call(thisp, value, i++, self)) {
                self._iterationObject.value = value;
            }
            else {
                self._iterationObject.done = true;
                self._iterationObject.value = void 0;
            }
        }
        return self._iterationObject;
    });

};

Iterator.prototype.zipIterator = function () {
    return Iterator.unzip(
        Array.prototype.concat.apply(this, arguments)
    );
};

Iterator.prototype.enumerateIterator = function (start) {
    return Iterator.count(start).zipIterator(this);
};

// creates an iterator for Array and String
Iterator.iterate = function (iterable) {
    var start;
    start = 0;
    return new Iterator(function () {
        // advance to next owned entry
        if (typeof iterable === "object") {
            while (!(start in iterable)) {
                // deliberately late bound
                if (start >= iterable.length) {
                    this._iterationObject.done = true;
                    this._iterationObject.value = void 0;
                    break;
                }
                else start += 1;
            }
        } else if (start >= iterable.length) {
            this._iterationObject.done = true;
            this._iterationObject.value = void 0;
        }

        if(!this._iterationObject.done) {
            this._iterationObject.value = iterable[start];
            start += 1;
        }
        return this._iterationObject;
    });
};

Iterator.cycle = function (cycle, times) {
    var next;
    if (arguments.length < 2)
        times = Infinity;
    //cycle = Iterator(cycle).toArray();
    return new Iterator(function () {
        var iteration, nextEntry;

        if(next) {
            nextEntry = next();
        }

        if(!next || nextEntry.done === true) {
            if (times > 0) {
                times--;
                iteration = Iterator.iterate(cycle);
                nextEntry = (next = iteration.next.bind(iteration))();
            }
            else {
                this._iterationObject.done = true;
                nextEntry = this._iterationObject;            }
        }
        return nextEntry;
    });
};

Iterator.concat = function (iterators) {
    iterators = Iterator(iterators);
    var next;
    return new Iterator(function (){
        var iteration, nextEntry;
        if(next) nextEntry = next();
        if(!nextEntry || nextEntry.done === true) {
            nextEntry = iterators.next();
            if(nextEntry.done === false) {
                iteration = Iterator(nextEntry.value);
                next = iteration.next.bind(iteration);
                return next();
            }
            else {
                return nextEntry;
            }
        }
        else return nextEntry;
    });
};

Iterator.unzip = function (iterators) {
    iterators = Iterator(iterators).map(Iterator);
    if (iterators.length === 0)
        return new Iterator([]);
    return new Iterator(function () {
        var stopped, nextEntry;
        var result = iterators.map(function (iterator) {
            nextEntry = iterator.next();
            if (nextEntry.done === true ) {
                stopped = true;
            }
            return nextEntry.value;
        });
        if (stopped) {
            this._iterationObject.done = true;
            this._iterationObject.value = void 0;
        }
        else {
            this._iterationObject.value = result;
        }
        return this._iterationObject;
    });
};

Iterator.zip = function () {
    return Iterator.unzip(
        Array.prototype.slice.call(arguments)
    );
};

Iterator.chain = function () {
    return Iterator.concat(
        Array.prototype.slice.call(arguments)
    );
};

Iterator.range = function (start, stop, step) {
    if (arguments.length < 3) {
        step = 1;
    }
    if (arguments.length < 2) {
        stop = start;
        start = 0;
    }
    start = start || 0;
    step = step || 1;
    return new Iterator(function () {
        if (start >= stop) {
            this._iterationObject.done = true;
            this._iterationObject.value = void 0;
        }
        var result = start;
        start += step;
        this._iterationObject.value = result;

        return this._iterationObject;
    });
};

Iterator.count = function (start, step) {
    return Iterator.range(start, Infinity, step);
};

Iterator.repeat = function (value, times) {
    return new Iterator.range(times).mapIterator(function () {
        return value;
    });
};
