/*! firebase-admin v4.2.1
    https://firebase.google.com/terms/ */
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var utils = require("../utils");
var error_1 = require("../utils/error");
/**
 * Parses a time stamp string or number and returns the corresponding date if valid.
 *
 * @param {any} time The unix timestamp string or number in milliseconds.
 * @return {Date} The corresponding date if valid.
 */
function parseDate(time) {
    try {
        var date = new Date(parseInt(time, 10));
        if (!isNaN(date.getTime())) {
            return date;
        }
    }
    catch (e) {
        // Do nothing. null will be returned.
    }
    return null;
}
/**
 * User metadata class that provides metadata information like user account creation
 * and last sign in time.
 *
 * @param {Object} response The server side response returned from the getAccountInfo
 *     endpoint.
 * @constructor
 */
var UserMetadata = (function () {
    function UserMetadata(response) {
        // Creation date should always be available but due to some backend bugs there
        // were cases in the past where users did not have creation date properly set.
        // This included legacy Firebase migrating project users and some anonymous users.
        // These bugs have already been addressed since then.
        utils.addReadonlyGetter(this, 'createdAt', parseDate(response.createdAt));
        utils.addReadonlyGetter(this, 'lastSignedInAt', parseDate(response.lastLoginAt));
    }
    /** @return {Object} The plain object representation of the user's metadata. */
    UserMetadata.prototype.toJSON = function () {
        return {
            lastSignedInAt: this.lastSignedInAt,
            createdAt: this.createdAt,
        };
    };
    return UserMetadata;
}());
exports.UserMetadata = UserMetadata;
/**
 * User info class that provides provider user information for different
 * Firebase providers like google.com, facebook.com, password, etc.
 *
 * @param {Object} response The server side response returned from the getAccountInfo
 *     endpoint.
 * @constructor
 */
var UserInfo = (function () {
    function UserInfo(response) {
        // Provider user id and provider id are required.
        if (!response.rawId || !response.providerId) {
            throw new error_1.FirebaseAuthError(error_1.AuthClientErrorCode.INTERNAL_ERROR, 'INTERNAL ASSERT FAILED: Invalid user info response');
        }
        utils.addReadonlyGetter(this, 'uid', response.rawId);
        utils.addReadonlyGetter(this, 'displayName', response.displayName);
        utils.addReadonlyGetter(this, 'email', response.email);
        utils.addReadonlyGetter(this, 'photoURL', response.photoUrl);
        utils.addReadonlyGetter(this, 'providerId', response.providerId);
    }
    /** @return {Object} The plain object representation of the current provider data. */
    UserInfo.prototype.toJSON = function () {
        return {
            uid: this.uid,
            displayName: this.displayName,
            email: this.email,
            photoURL: this.photoURL,
            providerId: this.providerId,
        };
    };
    return UserInfo;
}());
exports.UserInfo = UserInfo;
/**
 * User record class that defines the Firebase user object populated from
 * the Firebase Auth getAccountInfo response.
 *
 * @param {Object} response The server side response returned from the getAccountInfo
 *     endpoint.
 * @constructor
 */
var UserRecord = (function () {
    function UserRecord(response) {
        // The Firebase user id is required.
        if (!response.localId) {
            throw new error_1.FirebaseAuthError(error_1.AuthClientErrorCode.INTERNAL_ERROR, 'INTERNAL ASSERT FAILED: Invalid user response');
        }
        utils.addReadonlyGetter(this, 'uid', response.localId);
        utils.addReadonlyGetter(this, 'email', response.email);
        utils.addReadonlyGetter(this, 'emailVerified', !!response.emailVerified);
        utils.addReadonlyGetter(this, 'displayName', response.displayName);
        utils.addReadonlyGetter(this, 'photoURL', response.photoUrl);
        // If disabled is not provided, the account is enabled by default.
        utils.addReadonlyGetter(this, 'disabled', response.disabled || false);
        utils.addReadonlyGetter(this, 'metadata', new UserMetadata(response));
        var providerData = [];
        for (var _i = 0, _a = (response.providerUserInfo || []); _i < _a.length; _i++) {
            var entry = _a[_i];
            providerData.push(new UserInfo(entry));
        }
        utils.addReadonlyGetter(this, 'providerData', providerData);
    }
    /** @return {Object} The plain object representation of the user record. */
    UserRecord.prototype.toJSON = function () {
        var json = {
            uid: this.uid,
            email: this.email,
            emailVerified: this.emailVerified,
            displayName: this.displayName,
            photoURL: this.photoURL,
            disabled: this.disabled,
            // Convert metadata to json.
            metadata: this.metadata.toJSON(),
        };
        json.providerData = [];
        for (var _i = 0, _a = this.providerData; _i < _a.length; _i++) {
            var entry = _a[_i];
            // Convert each provider data to json.
            json.providerData.push(entry.toJSON());
        }
        return json;
    };
    return UserRecord;
}());
exports.UserRecord = UserRecord;
