/*! firebase-admin v4.2.1
    https://firebase.google.com/terms/ */
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var validator = require("./utils/validator");
var deep_copy_1 = require("./utils/deep-copy");
var credential_1 = require("./auth/credential");
var error_1 = require("./utils/error");
/**
 * Internals of a FirebaseApp instance.
 */
var FirebaseAppInternals = (function () {
    function FirebaseAppInternals(credential_) {
        this.credential_ = credential_;
        this.isDeleted_ = false;
        this.tokenListeners_ = [];
    }
    /**
     * Gets an auth token for the associated app.
     *
     * @param {boolean} forceRefresh Whether or not to force a token refresh.
     * @return {Promise<Object>} A Promise that will be fulfilled with the current or new token.
     */
    FirebaseAppInternals.prototype.getToken = function (forceRefresh) {
        var _this = this;
        var expired = this.cachedToken_ && this.cachedToken_.expirationTime < Date.now();
        if (this.cachedTokenPromise_ && !forceRefresh && !expired) {
            return this.cachedTokenPromise_
                .catch(function (error) {
                // Update the cached token promise to avoid caching errors. Set it to resolve with the
                // cached token if we have one (and return that promise since the token has still not
                // expired).
                if (_this.cachedToken_) {
                    _this.cachedTokenPromise_ = Promise.resolve(_this.cachedToken_);
                    return _this.cachedTokenPromise_;
                }
                // Otherwise, set the cached token promise to null so that it will force a refresh next
                // time getToken() is called.
                _this.cachedTokenPromise_ = null;
                // And re-throw the caught error.
                throw error;
            });
        }
        else {
            // Clear the outstanding token refresh timeout. This is a noop if the timeout is undefined.
            clearTimeout(this.tokenRefreshTimeout_);
            // this.credential_ may be an external class; resolving it in a promise helps us
            // protect against exceptions and upgrades the result to a promise in all cases.
            this.cachedTokenPromise_ = Promise.resolve(this.credential_.getAccessToken())
                .then(function (result) {
                // Since the developer can provide the credential implementation, we want to weakly verify
                // the return type until the type is properly exported.
                if (!validator.isNonNullObject(result) ||
                    typeof result.expires_in !== 'number' ||
                    typeof result.access_token !== 'string') {
                    throw new error_1.FirebaseAppError(error_1.AppErrorCodes.INVALID_CREDENTIAL, "Invalid access token generated: \"" + JSON.stringify(result) + "\". Valid access " +
                        'tokens must be an object with the "expires_in" (number) and "access_token" ' +
                        '(string) properties.');
                }
                var token = {
                    accessToken: result.access_token,
                    expirationTime: Date.now() + (result.expires_in * 1000),
                };
                var hasAccessTokenChanged = (_this.cachedToken_ && _this.cachedToken_.accessToken !== token.accessToken);
                var hasExpirationChanged = (_this.cachedToken_ && _this.cachedToken_.expirationTime !== token.expirationTime);
                if (!_this.cachedToken_ || hasAccessTokenChanged || hasExpirationChanged) {
                    _this.cachedToken_ = token;
                    _this.tokenListeners_.forEach(function (listener) {
                        listener(token.accessToken);
                    });
                }
                // Establish a timeout to proactively refresh the token every minute starting at five
                // minutes before it expires. Once a token refresh succeeds, no further retries are
                // needed; if it fails, retry every minute until the token expires (resulting in a total
                // of four retries: at 4, 3, 2, and 1 minutes).
                var refreshTimeInSeconds = (result.expires_in - (5 * 60));
                var numRetries = 4;
                // In the rare cases the token is short-lived (that is, it expires in less than five
                // minutes from when it was fetched), establish the timeout to refresh it after the
                // current minute ends and update the number of retries that should be attempted before
                // the token expires.
                if (refreshTimeInSeconds <= 0) {
                    refreshTimeInSeconds = result.expires_in % 60;
                    numRetries = Math.floor(result.expires_in / 60) - 1;
                }
                // The token refresh timeout keeps the Node.js process alive, so only create it if this
                // instance has not already been deleted.
                if (numRetries && !_this.isDeleted_) {
                    _this.setTokenRefreshTimeout(refreshTimeInSeconds * 1000, numRetries);
                }
                return token;
            })
                .catch(function (error) {
                var errorMessage = (typeof error === 'string') ? error : error.message;
                errorMessage = 'Credential implementation provided to initializeApp() via the ' +
                    '"credential" property failed to fetch a valid Google OAuth2 access token with the ' +
                    ("following error: \"" + errorMessage + "\".");
                if (errorMessage.indexOf('invalid_grant') !== -1) {
                    errorMessage += ' There are two likely causes: (1) your server time is not properly ' +
                        'synced or (2) your certificate key file has been revoked. To solve (1), re-sync the ' +
                        'time on your server. To solve (2), make sure the key ID for your key file is still ' +
                        'present at https://console.firebase.google.com/iam-admin/serviceaccounts/project. If ' +
                        'not, generate a new key file at ' +
                        'https://console.firebase.google.com/project/_/settings/serviceaccounts/adminsdk.';
                }
                throw new error_1.FirebaseAppError(error_1.AppErrorCodes.INVALID_CREDENTIAL, errorMessage);
            });
            return this.cachedTokenPromise_;
        }
    };
    /**
     * Adds a listener that is called each time a token changes.
     *
     * @param {function(string)} listener The listener that will be called with each new token.
     */
    FirebaseAppInternals.prototype.addAuthTokenListener = function (listener) {
        this.tokenListeners_.push(listener);
        if (this.cachedToken_) {
            listener(this.cachedToken_.accessToken);
        }
    };
    /**
     * Removes a token listener.
     *
     * @param {function(string)} listener The listener to remove.
     */
    FirebaseAppInternals.prototype.removeAuthTokenListener = function (listener) {
        this.tokenListeners_ = this.tokenListeners_.filter(function (other) { return other !== listener; });
    };
    /**
     * Deletes the FirebaseAppInternals instance.
     */
    FirebaseAppInternals.prototype.delete = function () {
        this.isDeleted_ = true;
        // Clear the token refresh timeout so it doesn't keep the Node.js process alive.
        clearTimeout(this.tokenRefreshTimeout_);
    };
    /**
     * Establishes timeout to refresh the Google OAuth2 access token used by the SDK.
     *
     * @param {number} delayInMilliseconds The delay to use for the timeout.
     * @param {number} numRetries The number of times to retry fetching a new token if the prior fetch
     *   failed.
     */
    FirebaseAppInternals.prototype.setTokenRefreshTimeout = function (delayInMilliseconds, numRetries) {
        var _this = this;
        this.tokenRefreshTimeout_ = setTimeout(function () {
            _this.getToken(/* forceRefresh */ true)
                .catch(function (error) {
                // Ignore the error since this might just be an intermittent failure. If we really cannot
                // refresh the token, an error will be logged once the existing token expires and we try
                // to fetch a fresh one.
                if (numRetries > 0) {
                    _this.setTokenRefreshTimeout(60 * 1000, numRetries - 1);
                }
            });
        }, delayInMilliseconds);
    };
    return FirebaseAppInternals;
}());
exports.FirebaseAppInternals = FirebaseAppInternals;
/**
 * Global context object for a collection of services using a shared authentication state.
 */
var FirebaseApp = (function () {
    function FirebaseApp(options, name, firebaseInternals_) {
        var _this = this;
        this.firebaseInternals_ = firebaseInternals_;
        this.services_ = {};
        this.isDeleted_ = false;
        this.name_ = name;
        this.options_ = deep_copy_1.deepCopy(options);
        if (typeof this.options_ !== 'object' || this.options_ === null) {
            // Ensure the options are a non-null object
            this.options_ = {};
        }
        var hasCredential = ('credential' in this.options_);
        var hasServiceAccount = ('serviceAccount' in this.options_);
        var errorMessage;
        if (!hasCredential && !hasServiceAccount) {
            errorMessage = 'Options must be an object containing at least a "credential" property.';
        }
        else if (hasCredential && hasServiceAccount) {
            errorMessage = 'Options cannot specify both the "credential" and "serviceAccount" properties.';
        }
        // TODO(jwenger): NEXT MAJOR RELEASE - throw error if the "credential" property is not specified
        if (hasServiceAccount) {
            var serviceAccount = this.options_.serviceAccount;
            var serviceAccountIsString = (typeof serviceAccount === 'string');
            var serviceAccountIsNonNullObject = (typeof serviceAccount === 'object' && serviceAccount !== null);
            if (!serviceAccountIsString && !serviceAccountIsNonNullObject) {
                errorMessage = 'The "serviceAccount" property must be a string representing the file path to ' +
                    'a key file or an object representing the contents of a key file.';
            }
        }
        else if (hasCredential) {
            var credential = this.options_.credential;
            if (typeof credential !== 'object' || credential === null || typeof credential.getAccessToken !== 'function') {
                errorMessage = 'The "credential" property must be an object which implements the Credential interface.';
            }
        }
        if (typeof errorMessage !== 'undefined') {
            throw new error_1.FirebaseAppError(error_1.AppErrorCodes.INVALID_APP_OPTIONS, "Invalid Firebase app options passed as the first argument to initializeApp() for the " +
                ("app named \"" + this.name_ + "\". " + errorMessage));
        }
        // TODO(jwenger): NEXT MAJOR RELEASE - remove "serviceAccount" property deprecation warning and
        // relevant error handling above
        if (hasServiceAccount) {
            /* tslint:disable:no-console */
            console.warn('WARNING: The "serviceAccount" property specified in the first argument to initializeApp() ' +
                'is deprecated and will be removed in the next major version. You should instead use the ' +
                '"credential" property.');
            /* tslint:enable:no-console */
            this.options_.credential = new credential_1.CertCredential(this.options_.serviceAccount);
        }
        Object.keys(firebaseInternals_.serviceFactories).forEach(function (serviceName) {
            // Defer calling createService() until the service is accessed
            _this[serviceName] = _this.getService_.bind(_this, serviceName);
        });
        this.INTERNAL = new FirebaseAppInternals(this.options_.credential);
    }
    /**
     * Firebase services available off of a FirebaseApp instance. These are monkey-patched via
     * registerService(), but we need to include a dummy implementation to get TypeScript to
     * compile it without errors.
     */
    /* istanbul ignore next */
    FirebaseApp.prototype.auth = function () {
        throw new error_1.FirebaseAppError(error_1.AppErrorCodes.INTERNAL_ERROR, 'INTERNAL ASSERT FAILED: Firebase auth() service has not been registered.');
    };
    /* istanbul ignore next */
    FirebaseApp.prototype.database = function () {
        throw new error_1.FirebaseAppError(error_1.AppErrorCodes.INTERNAL_ERROR, 'INTERNAL ASSERT FAILED: Firebase database() service has not been registered.');
    };
    /* istanbul ignore next */
    FirebaseApp.prototype.messaging = function () {
        throw new error_1.FirebaseAppError(error_1.AppErrorCodes.INTERNAL_ERROR, 'INTERNAL ASSERT FAILED: Firebase messaging() service has not been registered.');
    };
    Object.defineProperty(FirebaseApp.prototype, "name", {
        /**
         * Returns the name of the FirebaseApp instance.
         *
         * @returns {string} The name of the FirebaseApp instance.
         */
        get: function () {
            this.checkDestroyed_();
            return this.name_;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(FirebaseApp.prototype, "options", {
        /**
         * Returns the options for the FirebaseApp instance.
         *
         * @returns {FirebaseAppOptions} The options for the FirebaseApp instance.
         */
        get: function () {
            this.checkDestroyed_();
            return deep_copy_1.deepCopy(this.options_);
        },
        enumerable: true,
        configurable: true
    });
    /**
     * Deletes the FirebaseApp instance.
     *
     * @returns {Promise<void>} An empty Promise fulfilled once the FirebaseApp instance is deleted.
     */
    FirebaseApp.prototype.delete = function () {
        var _this = this;
        this.checkDestroyed_();
        this.firebaseInternals_.removeApp(this.name_);
        this.INTERNAL.delete();
        return Promise.all(Object.keys(this.services_).map(function (serviceName) {
            return _this.services_[serviceName].INTERNAL.delete();
        })).then(function () {
            _this.services_ = {};
            _this.isDeleted_ = true;
        });
    };
    /**
     * Returns the service instance associated with this FirebaseApp instance (creating it on demand
     * if needed).
     *
     * @param {string} serviceName The name of the service instance to return.
     * @return {FirebaseServiceInterface} The service instance with the provided name.
     */
    FirebaseApp.prototype.getService_ = function (serviceName) {
        this.checkDestroyed_();
        if (!(serviceName in this.services_)) {
            this.services_[serviceName] = this.firebaseInternals_.serviceFactories[serviceName](this, this.extendApp_.bind(this));
        }
        return this.services_[serviceName];
    };
    /**
     * Callback function used to extend an App instance at the time of service instance creation.
     */
    FirebaseApp.prototype.extendApp_ = function (props) {
        deep_copy_1.deepExtend(this, props);
    };
    /**
     * Throws an Error if the FirebaseApp instance has already been deleted.
     */
    FirebaseApp.prototype.checkDestroyed_ = function () {
        if (this.isDeleted_) {
            throw new error_1.FirebaseAppError(error_1.AppErrorCodes.APP_DELETED, "Firebase app named \"" + this.name_ + "\" has already been deleted.");
        }
    };
    return FirebaseApp;
}());
exports.FirebaseApp = FirebaseApp;
