/*global console*/
'use strict';

const crypto = require('crypto');
const BufferList = require('bl');
const { CONSTANTS } = require('../config/config');


/**
 * CrossPlatform CryptLib
   * This cross platform CryptLib uses AES 256 for encryption. This library can
   * be used for encryptoion and de-cryption of string on iOS, Android, Windows
   * and Node platform.
   * Features:
   * 1. 256 bit AES encryption
   * 2. Random IV generation.
   * 3. Provision for SHA256 hashing of key.
 */
class CryptLib {

    constructor() {
        this._maxKeySize = 32;
        this._maxIVSize = 16;
        this._algorithm = 'AES-256-CBC';
        this._charset = 'utf8';
        this._encoding = 'base64';
        this._hashAlgo = 'sha256';
        this._digestEncoding = 'hex';

        this._characterMatrixForRandomIVStringGeneration = [
            'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
            'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
            'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm',
            'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '-', '_'
        ];
    }

    /**
     * private function: _encryptDecrypt
     * encryptes or decrypts to or from text or encrypted text given an iv and key
     * @param  {string}  text        can be plain text or encrypted text
     * @param  {string}  key         the key used to encrypt or decrypt
     * @param  {string}  initVector  the initialization vector to encrypt or
     *                               decrypt
     * @param  {bool}    isEncrypt   true = encryption, false = decryption
     * @return {string}              encryted text or plain text
     */
    _encryptDecrypt(text, key, initVector, isEncrypt) {

        try {

            if (!text || !key) {
                throw 'cryptLib._encryptDecrypt: -> key and plain or encrypted text ' +
                'required';
            }

            let ivBl = new BufferList(),
                keyBl = new BufferList(),
                keyCharArray = key.split(''),
                ivCharArray = [],
                encryptor, decryptor, clearText;

            if (initVector && initVector.length > 0) {
                ivCharArray = initVector.split('');
            }

            for (let i = 0; i < this._maxIVSize; i++) {
                ivBl.append(ivCharArray.shift() || [null]);
            }

            for (let i = 0; i < this._maxKeySize; i++) {
                keyBl.append(keyCharArray.shift() || [null]);
            }

            if (isEncrypt) {
                encryptor = crypto.createCipheriv(this._algorithm, keyBl.toString(),
                    ivBl.toString());
                encryptor.setEncoding(this._encoding);
                encryptor.write(text);
                encryptor.end();
                return encryptor.read();
            }

            decryptor = crypto.createDecipheriv(this._algorithm, keyBl.toString(),
                ivBl.toString());
            let dec = decryptor.update(text, this._encoding, this._charset);
            dec += decryptor.final(this._charset);
            return dec;
        } catch (e) {
            return '';
        }
    }

    /**
     * private function: _isCorrectLength
     * checks if length is preset and is a whole number and > 0
     * @param  {int}  length
     * @return {bool}
    */
    _isCorrectLength(length) {
        return length && /^\d+$/.test(length) && parseInt(length, 10) !== 0
    }

    /**
     * generates random initialization vector given a length
     * @param  {int}  length  the length of the iv to be generated
     */
    generateRandomIV(length) {
        if (!this._isCorrectLength(length)) {
            throw 'cryptLib.generateRandomIV() -> needs length or in wrong format';
        }

        let randomBytes = crypto.randomBytes(length),
            _iv = [];

        for (let i = 0; i < length; i++) {
            let ptr = randomBytes[i] %
                this._characterMatrixForRandomIVStringGeneration.length;
            _iv[i] = this._characterMatrixForRandomIVStringGeneration[ptr];
        }
        return _iv.join('');
    }

    generateRandomIV16() {
        let randomBytes = crypto.randomBytes(16),
            _iv = [];

        for (let i = 0; i < 16; i++) {
            let ptr = randomBytes[i] %
                this._characterMatrixForRandomIVStringGeneration.length;
            _iv[i] = this._characterMatrixForRandomIVStringGeneration[ptr];
        }
        return _iv.join('');
    }

    /**
     * Creates a hash of a key using SHA-256 algorithm
     * @param  {string} key     the key that will be hashed
     * @param  {int}    length  the length of the SHA-256 hash
     * @return {string}         the output hash generated given a key and length
     */
    getHashSha256(key, length) {
        if (!key) {
            throw 'cryptLib.getHashSha256() -> needs key';
        }

        if (!this._isCorrectLength(length)) {
            throw 'cryptLib.getHashSha256() -> needs length or in wrong format';
        }

        return crypto.createHash(this._hashAlgo)
            .update(key)
            .digest(this._digestEncoding)
            .substring(0, length);
    }

    /**
     * encryptes plain text given a key and initialization vector
     * @param  {string}  text        can be plain text or encrypted text
     * @param  {string}  key         the key used to encrypt or decrypt
     * @param  {string}  initVector  the initialization vector to encrypt or
     *                               decrypt
     * @return {string}              encryted text or plain text
     */
    encrypt(plainText, key, initVector) {
        return this._encryptDecrypt(plainText, key, initVector, true);
    }

    /**
     * decrypts encrypted text given a key and initialization vector
     * @param  {string}  text        can be plain text or encrypted text
     * @param  {string}  key         the key used to encrypt or decrypt
     * @param  {string}  initVector  the initialization vector to encrypt or
     *                               decrypt
     * @return {string}              encryted text or plain text
     */
    decrypt(encryptedText, key, initVector) {
        return this._encryptDecrypt(encryptedText, key, initVector, false);
    }

    async encryptPlainTextWithRandomIV(plainText) {
        return this._encryptDecrypt(this.generateRandomIV16() + plainText, this.getHashSha256(CONSTANTS.AES_SECRET_KEY, 32), this.generateRandomIV16(), true);
    }

    async decryptCipherTextWithRandomIV(cipherText) {
        let out = this._encryptDecrypt(cipherText, this.getHashSha256(CONSTANTS.AES_SECRET_KEY, 32), this.generateRandomIV16(), false);
        return out.substring(16, out.length);
    }

    async aesDecrypt(input, arrayInput) {
        
        await this.asyncForEach(Object.keys(input), async (inputElement) => {
            // if need to check on this key or not
            if (arrayInput.includes(inputElement)) {
                // if that key is an array need to check again for internal element
                if (Array.isArray(input[inputElement])) {
                    await this.asyncForEach(input[inputElement], async (nestedElement, index) => {
                        if (nestedElement && typeof nestedElement === "object") {
                            await this.aesDecrypt(nestedElement, arrayInput)
                        } else {

                            input[inputElement][index] = await this.decryptCipherTextWithRandomIV(nestedElement);
                        }
                    })
                } else if (input[inputElement] && typeof input[inputElement] === "object") {
                    await this.aesDecrypt(input[inputElement], arrayInput);
                } else {
                    input[inputElement] = await this.decryptCipherTextWithRandomIV(input[inputElement]);
                }
            }
        })
    }

    async aesEncrypt(input, arrayInput) {
        await this.asyncForEach(Object.keys(input), async (inputElement) => {
            // if need to check on this key or not
            if (arrayInput.includes(inputElement)) {
                // if that key is an array need to check again for internal element
                if (Array.isArray(input[inputElement])) {
                    await this.asyncForEach(input[inputElement], async (nestedElement, index) => {
                        if (nestedElement && typeof nestedElement === "object") {
                            await this.aesEncrypt(nestedElement, arrayInput)
                        } else {
                            input[inputElement][index] = await this.encryptPlainTextWithRandomIV(nestedElement);
                        }
                    })
                } else if (input[inputElement] && typeof input[inputElement] === "object") {
                    await this.aesEncrypt(input[inputElement], arrayInput);
                } else {
                    input[inputElement] = await this.encryptPlainTextWithRandomIV(input[inputElement]);
                }

            }

        })

    }

    async bulkParseInt(input, arrayInput) {
        
        await this.asyncForEach(Object.keys(input), async (inputElement) => {
            // if need to check on this key or not
            if (arrayInput.includes(inputElement)) {
                // if that key is an array need to check again for internal element
                if (Array.isArray(input[inputElement])) {
                    await this.asyncForEach(input[inputElement], async (nestedElement, index) => {
                        if (nestedElement && typeof nestedElement === "object") {
                            await this.aesDecrypt(nestedElement, arrayInput)
                        } else {

                            input[inputElement][index] =  typeof parseInt(nestedElement);
                        }
                    })
                } else if (input[inputElement] && typeof input[inputElement] === "object") {
                    await this.aesDecrypt(input[inputElement], arrayInput);
                } else {
                    input[inputElement] =  parseInt(input[inputElement]);
                }
            }
        })
    }

    /**
     * Method: asyncForEach
     * Purpose: async ForEach loop
     */
    async asyncForEach(array, callback) {
        for (let index = 0; index < array.length; index++) {
            await callback(array[index], index, array);
        }
    }


}

module.exports = new CryptLib();